// palwiz.c - copyright 2005-2007 Bruce Tomlin

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
//#include "asmcfg.h"
//#include "asmguts.h"

#define versionName "Simple PAL/GAL compiler"
#define copyright "Copyright 2005-2007 Bruce Tomlin"

// integer types
//#include <stdint.h>
typedef unsigned int  u_int; // only used for EvalNum() results

// boolean type
#if 0 // for ANSI C
typedef unsigned char bool;
const bool false = 0;
const bool true  = 1;
#else // for C99
#include <stdbool.h>
#endif

// generic string type
typedef char Str255[256];


// ==============================================================


enum {
    MAXPINS  = 28+2+1,  // max pins + max phantom pins + 1 (so one-based subscripts can be used)
    MAXTERMS =    132,  // maximum number of fuse map rows
    MAXCOLS  =     44,  // maximum number of fuse map columns
    MAXFUSES =   7000,  // maximum number of fuses (22CV10 uses 6432 fuses)
    MAXOPTS  =      2,  // maximum number of option fuses per output pin or max chip-wide option fuses
    MAXTESTS =   9999,  // maximum number of test vectors (too lazy to use a linked list)
};


// ==============================================================
// chip data structure

// values for pininfo.outOE
enum outOE {
    no_OE  = 0,
    has_OE = 1,
    opt_OE = 2,
};

// values for pininfo.inFuse
enum inFuse {
    inFuse_GND = -1,
    inFuse_VCC = -2,
    inFuse_CLK = -3,
    inFuse_OE  = -4,
};

// bit-mapped values for pininfo.pinModes index, results in 5 combinations
enum {
    M_XOR    = 1,  // XOR output pin
    M_REG    = 2,  // registered
    M_INPUT  = 4,  // dedicated input
};

// --------------------------------------------------------------
// static data for each pin in a chip
typedef struct pininfo {
    int     outOE;              // list of output enable stats per pin
                        // 0 = no OE row, 1 = output pin has an OE row, 2 = optional OE row
                        // negative = fixed OE input pin, outOE[0] is used for 16V8R/20V8R
    int     inFuse;             // fuse map column of input pin, -1 GND, -2 VCC
    int     outFuse;            // first fuse for output pin
    int     outRows;            // number of rows for output pin
    int     pinModes;           // allowable modes for output pins
    int     optFuses[MAXOPTS];  // option fuse numbers
} pininfo_t;

// --------------------------------------------------------------
// static data for a chip type
typedef struct chipinfo {
    const char *chipName;   // name of chip type
    // FIXME: need a way to have forward reference chip types
    int     numCols;        // number of columns in term matrix
    int     numRows;        // number of rows in term matrix
    int     devFuses;       // total number of fuses in device
    int     devPins;        // number of pins on device
    int     phantomPins;    // devPins + number of phantom pins
    int     sigFuse;        // first signature fuse
    int     numSigFuses;    // number of signature fuses
    int     ptdBase;        // first PTD (product term disable) fuse, one per row

    const int *skipLine;    // list of fuse IDs at which to newline, zero terminated
    const int *modes;       // bit-mapped option pin combinations for each mode (0..4)
                            // indexed by pinMode, set to -1 if not allowed on this pin
    const int *addFuses;    // additional fuses to set for chip configuration
    const pininfo_t *pins;  // data per pin
} chipinfo_t;

const chipinfo_t *chip;     // info for currently selected chip type
char    col2pin[MAXCOLS];   // reverse lookup of an input pin from a column

// --------------------------------------------------------------
// data for the chip accumulated from the input file
bool    fuseMap[MAXFUSES];  // fuse map
char    *pinName[MAXPINS];  // pointer to pin name, null until assigned
bool    pinPhase[MAXPINS];  // 1=active high, 0=active low
bool    hasOE[MAXPINS];     // pin is fixed has_OE or an opt_OE pin has been used
char    nextRow[MAXPINS];   // next unused row for output pin
int     pinMode[MAXPINS];   // current mode for output pins
bool    outDef[MAXPINS];    // TRUE if output pin defined
bool    oeDef[MAXPINS];     // TRUE if output pin OE defined
char    *tests[MAXTESTS];   // test vectors
int     numTests;           // number of test vectors
int     defFuse;            // default fuse state
bool    secFuse;            // TRUE to blow security fuse


// ==============================================================
// --------------------------------------------------------------

const char      *progname;      // pointer to argv[0]

Str255          line;           // Current line from input file
char            *linePtr;       // pointer into current line for GetWord()
int             linenum;        // current line number in main source file
int             curToken;       // token value of current input word
Str255          word;           // current input word

//  Command line parameters
Str255          cl_SrcName;     // Source file name

FILE            *source;        // source input file
FILE            *object;        // object output file (set to stdout)


// ==============================================================
// chip data


// --------------------------------------------------------------
// 26CV12, global reset on phantom pin 29, global preset on pin 30
// modes:
//      S0=0 S1=0 - reg, active low
//      S0=1 S1=0 - reg, active high
//      S0=0 S1=1 - combo, active low
//      S0=1 S1=1 - combo, active high

// NOTE: the 26CV12 and 26V12 are different chips!
//       the 26V12 has extra option fuses (S2/S3) per macrocell
//           S2 selects a second clock input from pin 4
//           S3 selects a buried feedback mode

static const int chip_26CV12_skipLine[] = {
      52,  520,  988, 1456, 1924, 2496, 3172, 3848,
    4420, 4888, 5356, 5824, 6292, 6344, 6368,
    0
};

static const int chip_26CV12_modes[] = {
    3, // 0: combH - S0=1 S1=1
    2, // 1: combL - S0=0 S1=1
    1, // 2: regH  - S0=1 S1=0
    0, // 3: regL  - S0=0 S1=0
   -1, // 4: input - not allowed (use OE instead)
};

static const pininfo_t chip_26CV12_pins[] = {
        // outOE inFuse outFuse outRows pinModes optFuses
         { has_OE, -1,   -1, }, // pin 0          S0  S1
/*  1 */ {     -1,  0,   -1,  0,           0, {            }, },
/*  2 */ {     -1,  4,   -1,  0,           0, {            }, },
/*  3 */ {     -1,  8,   -1,  0,           0, {            }, },
/*  4 */ {     -1, 12,   -1,  0,           0, {            }, },
/*  5 */ {     -1, 16,   -1,  0,           0, {            }, },
/*  6 */ {     -1, 20,   -1,  0,           0, {            }, },
/*  7 */ {     -1, -2,   -1,  0,           0, {            }, }, // VCC
/*  8 */ {     -1, 24,   -1,  0,           0, {            }, },
/*  9 */ {     -1, 28,   -1,  0,           0, {            }, },
/* 10 */ {     -1, 32,   -1,  0,           0, {            }, },
/* 11 */ {     -1, 36,   -1,  0,           0, {            }, },
/* 12 */ {     -1, 40,   -1,  0,           0, {            }, },
/* 13 */ {     -1, 44,   -1,  0,           0, {            }, },
/* 14 */ {     -1, 48,   -1,  0,           0, {            }, },
/* 15 */ { has_OE, 50, 5824,  9, M_XOR|M_REG, { 6366, 6367 }, },
/* 16 */ { has_OE, 46, 5356,  9, M_XOR|M_REG, { 6366, 6365 }, },
/* 17 */ { has_OE, 42, 4888,  9, M_XOR|M_REG, { 6364, 6363 }, },
/* 18 */ { has_OE, 38, 4420,  9, M_XOR|M_REG, { 6362, 6361 }, },
/* 19 */ { has_OE, 34, 3848, 11, M_XOR|M_REG, { 6358, 6359 }, },
/* 20 */ { has_OE, 30, 3172, 13, M_XOR|M_REG, { 6356, 6357 }, },
/* 21 */ {     -1, -1,   -1,  0,           0, {            }, }, // GND
/* 22 */ { has_OE, 26, 2496, 13, M_XOR|M_REG, { 6354, 6355 }, },
/* 23 */ { has_OE, 22, 1924, 11, M_XOR|M_REG, { 6352, 6353 }, },
/* 24 */ { has_OE, 18, 1456,  9, M_XOR|M_REG, { 6350, 6351 }, },
/* 25 */ { has_OE, 14,  988,  9, M_XOR|M_REG, { 6348, 6349 }, },
/* 26 */ { has_OE, 10,  520,  9, M_XOR|M_REG, { 6346, 6437 }, },
/* 27 */ { has_OE,  6,   52,  9, M_XOR|M_REG, { 6344, 6345 }, },
/* 28 */ {     -1,  2,   -1,  0,           0, {            }, },
/* 29 */ {  no_OE,  0,    0,  1,           0, {            }, }, // RESET
/* 30 */ {  no_OE,  0, 6292,  1,           0, {            }, }, // PRESET
};

static const chipinfo_t chip_26CV12 = {
    .chipName    = "26CV12",
    .numCols     =   52,
    .numRows     =  122,
    .devFuses    = 6432,
    .devPins     =   28,
    .phantomPins =   30,
    .sigFuse     = 6368,
    .numSigFuses =   64,
    .ptdBase     =   -1, // no product term disable fuses
    .skipLine    = &chip_26CV12_skipLine[0],
    .modes       = &chip_26CV12_modes[0],
    .addFuses    = NULL,
    .pins        = &chip_26CV12_pins[0]
};

// --------------------------------------------------------------
// 22V10, global reset on phantom pin 25, global preset on pin 26
// modes:
//      S0=0 S1=0 - reg, active low
//      S0=1 S1=0 - reg, active high
//      S0=0 S1=1 - combo, active low
//      S0=1 S1=1 - combo, active high

static const int chip_22V10_skipLine[] = {
      44,  440,  924, 1496, 2156, 2904, 3652, 4312,
    4884, 5368, 5764, 5808, 5828,
    0
};

static const int chip_22V10_modes[] = {
    3, // 0: combH - S0=1 S1=1
    2, // 1: combL - S0=0 S1=1
    1, // 2: regH  - S0=1 S1=0
    0, // 3: regL  - S0=0 S1=0
   -1, // 4: input - not allowed (use OE instead)
};

static const pininfo_t chip_22V10_pins[] = {
        // outOE inFuse outFuse outRows pinModes optFuses
         { has_OE, -1,   -1, }, // pin 0          S0  S1
/*  1 */ {     -1,  0,   -1,  0,           0, {            }, },
/*  2 */ {     -1,  4,   -1,  0,           0, {            }, },
/*  3 */ {     -1,  8,   -1,  0,           0, {            }, },
/*  4 */ {     -1, 12,   -1,  0,           0, {            }, },
/*  5 */ {     -1, 16,   -1,  0,           0, {            }, },
/*  6 */ {     -1, 20,   -1,  0,           0, {            }, },
/*  7 */ {     -1, 24,   -1,  0,           0, {            }, },
/*  8 */ {     -1, 28,   -1,  0,           0, {            }, },
/*  9 */ {     -1, 32,   -1,  0,           0, {            }, },
/* 10 */ {     -1, 36,   -1,  0,           0, {            }, },
/* 11 */ {     -1, 40,   -1,  0,           0, {            }, },
/* 12 */ {     -1, -1,   -1,  0,           0, {            }, }, // GND
/* 13 */ {     -1, 42,   -1,  0,           0, {            }, },
/* 14 */ { has_OE, 38, 5368,  9, M_XOR|M_REG, { 5826, 5827 }, },
/* 15 */ { has_OE, 34, 4884, 11, M_XOR|M_REG, { 5824, 5825 }, },
/* 16 */ { has_OE, 30, 4312, 13, M_XOR|M_REG, { 5822, 5823 }, },
/* 17 */ { has_OE, 26, 3652, 15, M_XOR|M_REG, { 5820, 5821 }, },
/* 18 */ { has_OE, 22, 2904, 17, M_XOR|M_REG, { 5818, 5819 }, },
/* 19 */ { has_OE, 18, 2156, 17, M_XOR|M_REG, { 5816, 5817 }, },
/* 20 */ { has_OE, 14, 1496, 15, M_XOR|M_REG, { 5814, 5815 }, },
/* 21 */ { has_OE, 10,  924, 13, M_XOR|M_REG, { 5812, 5813 }, },
/* 22 */ { has_OE,  6,  440, 11, M_XOR|M_REG, { 5810, 5811 }, },
/* 23 */ { has_OE,  2,   44,  9, M_XOR|M_REG, { 5808, 5809 }, },
/* 24 */ {     -1, -2,   -1,  0,           0, {            }, }, // VCC
/* 25 */ {  no_OE,  0,    0,  1,           0, {            }, }, // RESET
/* 26 */ {  no_OE,  0, 5764,  1,           0, {            }, }, // PRESET
};

static const chipinfo_t chip_22V10 = {
    .chipName    = "22V10",
    .numCols     =   44,
    .numRows     =  132,
    .devFuses    = 5892,
    .devPins     =   24,
    .phantomPins =   26,
    .sigFuse     = 5828,
    .numSigFuses =   64,
    .ptdBase     =   -1, // no product term disable fuses
    .skipLine    = &chip_22V10_skipLine[0],
    .modes       = &chip_22V10_modes[0],
    .addFuses    = NULL,
    .pins        = &chip_22V10_pins[0],
};

// --------------------------------------------------------------
// 20V8, simple mode - pins 18 and 19 are output-only
// modes:
//    SYN=1 AC0=0 - simple mode
//      XOR=0 AC1=0 - feedback, active low
//      XOR=1 AC1=0 - feedback, active high
//      XOR=X AC1=1 - dedicated input pin

static const int chip_20V8_skipLine[] = {
     320,  640,  960, 1280, 1600, 1920, 2240, 2560,
    2568, 2632, 2640, 2704, 2705,
    0
};

static const int chip_20V8S_addFuses[] = {
    2704, // 2704 TRUE   SYN = 1
   -2705, // 2705 FALSE  AC0 = 0
    0
};

static const int chip_20V8S_modes[] = {
    1, // 0: combH - XOR=1 AC1=0
    0, // 1: combL - XOR=0 AC1=0
   -1, // 2: regH  - not allowed
   -1, // 3: regL  - not allowed
    3, // 4: input - XOR=X AC1=1
};

static const pininfo_t chip_20V8S_pins[] = {
        // outOE inFuse outFuse outRows pinModes optFuses
         {  no_OE, -1,   -1, }, // pin 0         XOR  AC1
/*  1 */ {     -1,  2,   -1,  0,           0, {            }, },
/*  2 */ {     -1,  0,   -1,  0,           0, {            }, },
/*  3 */ {     -1,  4,   -1,  0,           0, {            }, },
/*  4 */ {     -1,  8,   -1,  0,           0, {            }, },
/*  5 */ {     -1, 12,   -1,  0,           0, {            }, },
/*  6 */ {     -1, 16,   -1,  0,           0, {            }, },
/*  7 */ {     -1, 20,   -1,  0,           0, {            }, },
/*  8 */ {     -1, 24,   -1,  0,           0, {            }, },
/*  9 */ {     -1, 28,   -1,  0,           0, {            }, },
/* 10 */ {     -1, 32,   -1,  0,           0, {            }, },
/* 11 */ {     -1, 36,   -1,  0,           0, {            }, },
/* 12 */ {     -1, -1,   -1,  0,           0, {            }, }, // GND
/* 13 */ {     -1, 38,   -1,  0,           0, {            }, },
/* 14 */ {     -1, 34,   -1,  0,           0, {            }, },
/* 15 */ {  no_OE, 30, 2240,  8,       M_XOR, { 2567, 2639 }, },
/* 16 */ {  no_OE, 26, 1920,  8,       M_XOR, { 2566, 2638 }, },
/* 17 */ {  no_OE, 22, 1600,  8,       M_XOR, { 2565, 2637 }, },
/* 18 */ {  no_OE, -3, 1280,  8,       M_XOR, { 2564, 2636 }, }, // no feedback
/* 19 */ {  no_OE, -4,  960,  8,       M_XOR, { 2563, 2635 }, }, // no feedback
/* 20 */ {  no_OE, 18,  640,  8,       M_XOR, { 2562, 2634 }, },
/* 21 */ {  no_OE, 14,  320,  8,       M_XOR, { 2561, 2633 }, },
/* 22 */ {  no_OE, 10,    0,  8,       M_XOR, { 2560, 2632 }, },
/* 23 */ {     -1,  6,   -1,  0,           0, {            }, },
/* 24 */ {     -1, -2,   -1,  0,           0, {            }, }, // VCC
};

static const chipinfo_t chip_20V8S = {
    .chipName    = "20V8S",
    .numCols     =   40,
    .numRows     =   64,
    .devFuses    = 2706,
    .devPins     =   24,
    .phantomPins =   24, // no phantom pins
    .sigFuse     = 2568,
    .numSigFuses =   64,
    .ptdBase     = 2640, // product term disable fuses, one per row
    .skipLine    = &chip_20V8_skipLine[0],
    .modes       = &chip_20V8S_modes[0],
    .addFuses    = &chip_20V8S_addFuses[0],
    .pins        = &chip_20V8S_pins[0],
};

// --------------------------------------------------------------
// 20V8, complex mode - pins 15 and 22 are output-only
// modes:
//    SYN=1 AC0=0 - simple mode
//      XOR=0 AC1=0 - feedback, active low
//      XOR=1 AC1=0 - feedback, active high
//      XOR=X AC1=1 - dedicated input pin

static const int chip_20V8C_addFuses[] = {
    2704, // 2704 TRUE   SYN = 1
    2705, // 2705 TRUE   AC0 = 1
    0
};

static const int chip_20V8C_modes[] = {
    3, // 0: combH - XOR=1 AC1=1
    2, // 1: combL - XOR=0 AC1=1
   -1, // 2: regH  - not allowed
   -1, // 3: regL  - not allowed
   -1, // 4: input - not allowed
};

static const pininfo_t chip_20V8C_pins[] = {
        // outOE inFuse outFuse outRows pinModes optFuses
         { has_OE, -1,   -1, }, // pin 0         XOR  AC1
/*  1 */ {     -1,  2,   -1,  0,           0, {            }, },
/*  2 */ {     -1,  0,   -1,  0,           0, {            }, },
/*  3 */ {     -1,  4,   -1,  0,           0, {            }, },
/*  4 */ {     -1,  8,   -1,  0,           0, {            }, },
/*  5 */ {     -1, 12,   -1,  0,           0, {            }, },
/*  6 */ {     -1, 16,   -1,  0,           0, {            }, },
/*  7 */ {     -1, 20,   -1,  0,           0, {            }, },
/*  8 */ {     -1, 24,   -1,  0,           0, {            }, },
/*  9 */ {     -1, 28,   -1,  0,           0, {            }, },
/* 10 */ {     -1, 32,   -1,  0,           0, {            }, },
/* 11 */ {     -1, 36,   -1,  0,           0, {            }, },
/* 12 */ {     -1, -1,   -1,  0,           0, {            }, }, // GND
/* 13 */ {     -1, 38,   -1,  0,           0, {            }, },
/* 14 */ {     -1, 34,   -1,  0,           0, {            }, },
/* 15 */ { has_OE, -3, 2240,  8,       M_XOR, { 2567, 2639 }, }, // no feedback
/* 16 */ { has_OE, 30, 1920,  8,       M_XOR, { 2566, 2638 }, },
/* 17 */ { has_OE, 26, 1600,  8,       M_XOR, { 2565, 2637 }, },
/* 18 */ { has_OE, 22, 1280,  8,       M_XOR, { 2564, 2636 }, },
/* 19 */ { has_OE, 18,  960,  8,       M_XOR, { 2563, 2635 }, },
/* 20 */ { has_OE, 14,  640,  8,       M_XOR, { 2562, 2634 }, },
/* 21 */ { has_OE, 10,  320,  8,       M_XOR, { 2561, 2633 }, },
/* 22 */ { has_OE, -4,    0,  8,       M_XOR, { 2560, 2632 }, }, // no feedback
/* 23 */ {     -1,  6,   -1,  0,           0, {            }, },
/* 24 */ {     -1, -2,   -1,  0,           0, {            }, }, // VCC
};

static const chipinfo_t chip_20V8C = {
    .chipName    = "20V8C",
    .numCols     =   40,
    .numRows     =   64,
    .devFuses    = 2706,
    .devPins     =   24,
    .phantomPins =   24, // no phantom pins
    .sigFuse     = 2568,
    .numSigFuses =   64,
    .ptdBase     = 2640, // product term disable fuses, one per row
    .skipLine    = &chip_20V8_skipLine[0],
    .modes       = &chip_20V8C_modes[0],
    .addFuses    = &chip_20V8C_addFuses[0],
    .pins        = &chip_20V8C_pins[0],
};

// --------------------------------------------------------------
// 20V8, registered mode
// modes:
//    SYN=0 AC0=1 - registered mode
//      XOR=0 AC1=0 - reg, active low
//      XOR=1 AC1=0 - reg, active high
//      XOR=0 AC1=1 - combo, active low
//      XOR=1 AC1=1 - combo, active high
//      pin 1 is global clock
//      pin 13 is global !OE


static const int chip_20V8R_addFuses[] = {
   -2704, // 2704 FALSE  SYN = 0
    2705, // 2705 TRUE   AC0 = 1
    0
};

static const int chip_20V8R_modes[] = {
   3, // 0: combH - XOR=1 AC1=1
   2, // 1: combL - XOR=0 AC1=1
   1, // 2: regH  - XOR=1 AC1=0
   0, // 3: regL  - XOR=0 AC1=0
  -1, // 4: not allowed (use OE instead)
};

static const pininfo_t chip_20V8R_pins[] = {
        // outOE inFuse outFuse outRows pinModes optFuses
         { opt_OE, -1,   -1, }, // pin 0         XOR  AC1
/*  1 */ {     -1, -3,   -1,  0,           0, {            }, }, // CLK
/*  2 */ {     -1,  0,   -1,  0,           0, {            }, },
/*  3 */ {     -1,  4,   -1,  0,           0, {            }, },
/*  4 */ {     -1,  8,   -1,  0,           0, {            }, },
/*  5 */ {     -1, 12,   -1,  0,           0, {            }, },
/*  6 */ {     -1, 16,   -1,  0,           0, {            }, },
/*  7 */ {     -1, 20,   -1,  0,           0, {            }, },
/*  8 */ {     -1, 24,   -1,  0,           0, {            }, },
/*  9 */ {     -1, 28,   -1,  0,           0, {            }, },
/* 10 */ {     -1, 32,   -1,  0,           0, {            }, },
/* 11 */ {     -1, 36,   -1,  0,           0, {            }, },
/* 12 */ {     -1, -1,   -1,  0,           0, {            }, }, // GND
/* 13 */ {     -1, -4,   -1,  0,           0, {            }, }, // OE
/* 14 */ {     -1, 38,   -1,  0,           0, {            }, },
/* 15 */ { opt_OE, 34, 2240,  8, M_XOR|M_REG, { 2567, 2639 }, },
/* 16 */ { opt_OE, 30, 1920,  8, M_XOR|M_REG, { 2566, 2638 }, },
/* 17 */ { opt_OE, 26, 1600,  8, M_XOR|M_REG, { 2565, 2637 }, },
/* 18 */ { opt_OE, 22, 1280,  8, M_XOR|M_REG, { 2564, 2636 }, },
/* 19 */ { opt_OE, 18,  960,  8, M_XOR|M_REG, { 2563, 2635 }, },
/* 20 */ { opt_OE, 14,  640,  8, M_XOR|M_REG, { 2562, 2634 }, },
/* 21 */ { opt_OE, 10,  320,  8, M_XOR|M_REG, { 2561, 2633 }, },
/* 22 */ { opt_OE,  6,    0,  8, M_XOR|M_REG, { 2560, 2632 }, },
/* 23 */ {     -1,  2,   -1,  0,           0, {            }, },
/* 24 */ {     -1, -2,   -1,  0,           0, {            }, }, // VCC
};

static const chipinfo_t chip_20V8R = {
    .chipName    = "20V8R",
    .numCols     =   40,
    .numRows     =   64,
    .devFuses    = 2706,
    .devPins     =   24,
    .phantomPins =   24, // no phantom pins
    .sigFuse     = 2568,
    .numSigFuses =   64,
    .ptdBase     = 2640, // product term disable fuses, one per row
    .skipLine    = &chip_20V8_skipLine[0],
    .modes       = &chip_20V8R_modes[0],
    .addFuses    = &chip_20V8R_addFuses[0],
    .pins        = &chip_20V8R_pins[0],
};

// --------------------------------------------------------------
// 16V8, simple mode
// modes:
//    SYN=1 AC0=0 - simple mode
//      XOR=0 AC1=0 - feedback, active low
//      XOR=1 AC1=0 - feedback, active high
//      XOR=X AC1=1 - dedicated input pin

static const int chip_16V8_skipLine[] = {
     256,  512,  768, 1024, 1280, 1536, 1792, 2048,
    2056, 2120, 2128, 2192, 2193,
    0
};

static const int chip_16V8S_addFuses[] = {
    2192, // 2192 TRUE   SYN = 1
   -2193, // 2193 FALSE  AC0 = 0
    0
};

static const int chip_16V8S_modes[] = {
    1, // 0: combH - XOR=1 AC1=0
    0, // 1: combL - XOR=0 AC1=0
   -1, // 2: regH  - not allowed
   -1, // 3: regL  - not allowed
    3, // 4: input - XOR=X AC1=1
};

static const pininfo_t chip_16V8S_pins[] = {
        // outOE inFuse outFuse outRows pinModes optFuses
         {  no_OE, -1,   -1, }, // pin 0         XOR  AC1
/*  1 */ {     -1,  2,   -1,  0,           0, {            }, },
/*  2 */ {     -1,  0,   -1,  0,           0, {            }, },
/*  3 */ {     -1,  4,   -1,  0,           0, {            }, },
/*  4 */ {     -1,  8,   -1,  0,           0, {            }, },
/*  5 */ {     -1, 12,   -1,  0,           0, {            }, },
/*  6 */ {     -1, 16,   -1,  0,           0, {            }, },
/*  7 */ {     -1, 20,   -1,  0,           0, {            }, },
/*  8 */ {     -1, 24,   -1,  0,           0, {            }, },
/*  9 */ {     -1, 28,   -1,  0,           0, {            }, },
/* 10 */ {     -1, -1,   -1,  0,           0, {            }, }, // GND
/* 11 */ {     -1, 30,   -1,  0,           0, {            }, },
/* 12 */ {  no_OE, 26, 1792,  8,       M_XOR, { 2055, 2127 }, },
/* 13 */ {  no_OE, 22, 1536,  8,       M_XOR, { 2054, 2126 }, },
/* 14 */ {  no_OE, 18, 1280,  8,       M_XOR, { 2053, 2125 }, },
/* 15 */ {  no_OE, -3, 1024,  8,       M_XOR, { 2052, 2124 }, }, // no feedback
/* 16 */ {  no_OE, -4,  768,  8,       M_XOR, { 2051, 2123 }, }, // no feedback
/* 17 */ {  no_OE, 14,  512,  8,       M_XOR, { 2050, 2122 }, },
/* 18 */ {  no_OE, 10,  256,  8,       M_XOR, { 2049, 2121 }, },
/* 19 */ {  no_OE,  6,    0,  8,       M_XOR, { 2048, 2120 }, },
/* 20 */ {     -1, -2,   -1,  0,           0, {            }, }, // VCC
};

static const chipinfo_t chip_16V8S = {
    .chipName    = "16V8S",
    .numCols     =   32,
    .numRows     =   64,
    .devFuses    = 2194,
    .devPins     =   20,
    .phantomPins =   20, // no phantom pins
    .sigFuse     = 2568,
    .numSigFuses =   64,
    .ptdBase     = 2128, // product term disable fuses, one per row
    .skipLine    = &chip_16V8_skipLine[0],
    .modes       = &chip_16V8S_modes[0],
    .addFuses    = &chip_16V8S_addFuses[0],
    .pins        = &chip_16V8S_pins[0],
};

// --------------------------------------------------------------
// 16V8, complex mode
// modes:
//    SYN=1 AC0=1 - complex mode
//      XOR=0 AC1=0 - reg, active low
//      XOR=1 AC1=0 - reg, active high
//      XOR=0 AC1=1 - combo, active low
//      XOR=1 AC1=1 - combo, active high

static const int chip_16V8C_addFuses[] = {
    2192, // 2192 TRUE   SYN = 1
    2193, // 2193 TRUE   AC0 = 1
    0
};

static const int chip_16V8C_modes[] = {
    3, // 0: combH - XOR=1 AC1=1
    2, // 1: combL - XOR=0 AC1=1
   -1, // 2: regH  - not allowed
   -1, // 3: regL  - not allowed
   -1, // 4: input - not allowed
};

static const pininfo_t chip_16V8C_pins[] = {
        // outOE inFuse outFuse outRows pinModes optFuses
         { has_OE, -1,   -1, }, // pin 0         XOR  AC1
/*  1 */ {     -1,  2,   -1,  0,           0, {            }, },
/*  2 */ {     -1,  0,   -1,  0,           0, {            }, },
/*  3 */ {     -1,  4,   -1,  0,           0, {            }, },
/*  4 */ {     -1,  8,   -1,  0,           0, {            }, },
/*  5 */ {     -1, 12,   -1,  0,           0, {            }, },
/*  6 */ {     -1, 16,   -1,  0,           0, {            }, },
/*  7 */ {     -1, 20,   -1,  0,           0, {            }, },
/*  8 */ {     -1, 24,   -1,  0,           0, {            }, },
/*  9 */ {     -1, 28,   -1,  0,           0, {            }, },
/* 10 */ {     -1, -1,   -1,  0,           0, {            }, }, // GND
/* 11 */ {     -1, 30,   -1,  0,           0, {            }, },
/* 12 */ { has_OE, 26, 1792,  8,       M_XOR, { 2055, 2127 }, }, // no feedback
/* 13 */ { has_OE, 22, 1536,  8,       M_XOR, { 2054, 2126 }, },
/* 14 */ { has_OE, 18, 1280,  8,       M_XOR, { 2053, 2125 }, },
/* 15 */ { has_OE, -3, 1024,  8,       M_XOR, { 2052, 2124 }, },
/* 16 */ { has_OE, -4,  768,  8,       M_XOR, { 2051, 2123 }, },
/* 17 */ { has_OE, 14,  512,  8,       M_XOR, { 2050, 2122 }, },
/* 18 */ { has_OE, 10,  256,  8,       M_XOR, { 2049, 2121 }, },
/* 19 */ { has_OE,  6,    0,  8,       M_XOR, { 2048, 2120 }, }, // no feedback
/* 20 */ {     -1, -2,   -1,  0,           0, {            }, }, // VCC
};

static const chipinfo_t chip_16V8C = {
    .chipName    = "16V8C",
    .numCols     =   32,
    .numRows     =   64,
    .devFuses    = 2194,
    .devPins     =   20,
    .phantomPins =   20, // no phantom pins
    .sigFuse     = 2568, // FIXME: confirm this! is this really same as 20V8?
    .numSigFuses =   64,
    .ptdBase     = 2128, // product term disable fuses, one per row
    .skipLine    = &chip_16V8_skipLine[0],
    .modes       = &chip_16V8C_modes[0],
    .addFuses    = &chip_16V8C_addFuses[0],
    .pins        = &chip_16V8C_pins[0],
};

// --------------------------------------------------------------
// 16V8, registered mode
// modes:
//    SYN=0 AC0=1 - registered mode
//      XOR=0 AC1=0 - reg, active low
//      XOR=1 AC1=0 - reg, active high
//      XOR=0 AC1=1 - combo, active low
//      XOR=1 AC1=1 - combo, active high
//      pin 1 is global clock
//      pin 13 is global !OE

static const int chip_16V8R_addFuses[] = {
   -2192, // 2192 FALSE  SYN = 0
    2193, // 2193 TRUE   AC0 = 1
    0
};

static const int chip_16V8R_modes[] = {
    3, // 0: combH - XOR=1 AC1=1
    2, // 1: combL - XOR=0 AC1=1
    1, // 2: regH  - XOR=1 AC1=0
    0, // 3: regL  - XOR=0 AC1=0
   -1, // 4: not allowed (use OE instead)
};

static const pininfo_t chip_16V8R_pins[] = {
        // outOE inFuse outFuse outRows pinModes optFuses
         { opt_OE, -1,   -1, }, // pin 0         XOR  AC1
/*  1 */ {     -1, -3,   -1,  0,           0, {            }, }, // CLK
/*  2 */ {     -1,  0,   -1,  0,           0, {            }, },
/*  3 */ {     -1,  4,   -1,  0,           0, {            }, },
/*  4 */ {     -1,  8,   -1,  0,           0, {            }, },
/*  5 */ {     -1, 12,   -1,  0,           0, {            }, },
/*  6 */ {     -1, 16,   -1,  0,           0, {            }, },
/*  7 */ {     -1, 20,   -1,  0,           0, {            }, },
/*  8 */ {     -1, 24,   -1,  0,           0, {            }, },
/*  9 */ {     -1, 28,   -1,  0,           0, {            }, },
/* 10 */ {     -1, -1,   -1,  0,           0, {            }, }, // GND
/* 11 */ {     -1, -4,   -1,  0,           0, {            }, }, // OE
/* 12 */ { opt_OE, 30, 1792,  8, M_XOR|M_REG, { 2055, 2127 }, },
/* 13 */ { opt_OE, 26, 1536,  8, M_XOR|M_REG, { 2054, 2126 }, },
/* 14 */ { opt_OE, 22, 1280,  8, M_XOR|M_REG, { 2053, 2125 }, },
/* 15 */ { opt_OE, 18, 1024,  8, M_XOR|M_REG, { 2052, 2124 }, },
/* 16 */ { opt_OE, 14,  768,  8, M_XOR|M_REG, { 2051, 2123 }, },
/* 17 */ { opt_OE, 10,  512,  8, M_XOR|M_REG, { 2050, 2122 }, },
/* 18 */ { opt_OE,  6,  256,  8, M_XOR|M_REG, { 2049, 2121 }, },
/* 19 */ { opt_OE,  2,    0,  8, M_XOR|M_REG, { 2048, 2120 }, },
/* 20 */ {     -1, -2,   -1,  0,           0, {            }, }, // VCC
};

static const chipinfo_t chip_16V8R = {
    .chipName    = "16V8R",
    .numCols     =   32,
    .numRows     =   64,
    .devFuses    = 2194,
    .devPins     =   20,
    .phantomPins =   20, // no phantom pins
    .sigFuse     = 2568,
    .numSigFuses =   64,
    .ptdBase     = 2128, // product term disable fuses, one per row
    .skipLine    = &chip_16V8_skipLine[0],
    .modes       = &chip_16V8R_modes[0],
    .addFuses    = &chip_16V8R_addFuses[0],
    .pins        = &chip_16V8R_pins[0],
};

// --------------------------------------------------------------

/*static*/ const chipinfo_t *chipList[] = {
    &chip_26CV12,   // 26CV12 (not 26V12!) - untested!
    &chip_22V10,    // 22V10
    &chip_20V8S,    // 20V8 in simple mode
    &chip_20V8C,    // 20V8 in complex mode
    &chip_20V8R,    // 20V8 in registered mode
    &chip_16V8S,    // 16V8 in simple mode
    &chip_16V8C,    // 16V8 in complex mode
    &chip_16V8R,    // 16V8 in registered mode
    NULL            // end of list
};


// ==============================================================
// error messages


void EatLine(void); // forward declaration


/*
 *  Error
 */

void Error(char *message)
{
    fprintf(stderr, "%s:%d:%s\n", cl_SrcName, linenum, message);

    EatLine(); // discard the rest of the current line
}


void Warning(char *message)
{
    fprintf(stderr, "%s:%d: *** Warning:  %s ***\n", cl_SrcName, linenum, message);
}


// ==============================================================
// string utilities


/*
 *  Debleft deblanks the string s from the left side
 */

void Debleft(char *s)
{
    char *p = s;

    while (*p == 9 || *p == ' ') {
        p++;
    }

    if (p != s) {
        while ((*s++ = *p++));
    }
}


/*
 *  Debright deblanks the string s from the right side
 */

void Debright(char *s)
{
    char *p = s + strlen(s);

    while (p > s && *--p == ' ') {
        *p = 0;
    }
}


/*
 *  Deblank removes blanks from both ends of the string s
 */

void Deblank(char *s)
{
    Debleft(s);
    Debright(s);
}


/*
 *  Uprcase converts string s to upper case
 */

void Uprcase(char *s)
{
    char *p = s;

    while ((*p = toupper(*p))) {
        p++;
    }
}


char *newstring(char *s)
{
    char *p;
    p = malloc(strlen(s) + 1);
    if (p) {
        strcpy(p, s);
    } else {
        Error("can't allocate new string");
    }
    return p;
}


// ==============================================================
// numeric evaluation


u_int EvalBin(char *binStr)
{

    bool    evalErr = false;
    u_int   binVal  = 0;

    int     c;
    while ((c = *binStr++)) {
        if (c < '0' || c > '1') {
            evalErr = true;
            break;
        } else {
            binVal = binVal * 2 + c - '0';
        }
    }

    if (evalErr) {
        binVal = 0;
        Error("Invalid binary number");
    }

   return binVal;
}


u_int EvalOct(char *octStr)
{
    bool    evalErr = false;
    u_int   octVal  = 0;

    int     c;
    while ((c = *octStr++)) {
        if (c < '0' || c > '7') {
            evalErr = true;
            break;
        } else {
            octVal = octVal * 8 + c - '0';
        }
    }

    if (evalErr) {
        octVal = 0;
        Error("Invalid octal number");
    }

   return octVal;
}


u_int EvalDec(char *decStr)
{
    bool    evalErr = false;
    u_int   decVal  = 0;

    int     c;
    while ((c = *decStr++)) {
        if (!isdigit(c)) {
            evalErr = true;
            break;
        } else {
            decVal = decVal * 10 + c - '0';
        }
    }

    if (evalErr) {
        decVal = 0;
        Error("Invalid decimal number");
    }

   return decVal;
}


u_int EvalHex(char *hexStr)
{
    bool    evalErr = false;
    u_int   hexVal  = 0;

    int     c;
    while ((c = *hexStr++)) {
        if (!isxdigit(c)) {
            evalErr = true;
            break;
        } else {
            if (c > '9') {
                c = toupper(c) - 'A' + 10;
            } else {
                c = c - '0';
            }
            hexVal = hexVal * 16 + c;
        }
    }

    if (evalErr) {
        hexVal = 0;
        Error("Invalid hexadecimal number");
    }

    return hexVal;
}


u_int EvalNum(char *word)
{
    // handle C-style 0xnnnn hexadecimal constants
    if (word[0] == '0') {
        if (toupper(word[1]) == 'X') {
            return EvalHex(word + 2);
        }
        // return EvalOct(word);    // 0nnn octal constants are in less demand, though
    }

    int val = strlen(word) - 1;
    switch (toupper(word[val])) {
        case 'B':
            word[val] = 0;
            val = EvalBin(word);
            break;

        case 'O':
            word[val] = 0;
            val = EvalOct(word);
            break;

        case 'D':
            word[val] = 0;
            val = EvalDec(word);
            break;

        case 'H':
            word[val] = 0;
            val = EvalHex(word);
            break;

        default:
            val = EvalDec(word);
            break;
    }

    return val;
}


// ==============================================================
// token handling


// returns 0 for end-of-line, -1 for alpha-numeric, else char value for non-alphanumeric
// converts the word to uppercase, too
int GetWord(char *word)
{
    word[0] = 0;

    // skip initial whitespace
    char c = *linePtr;
    while (c == 12 || c == '\t' || c == ' ') {
        c = *++linePtr;
    }

    // skip comments
    if (c == '/' && linePtr[1] == '/') {
        while (c) {
            c = *++linePtr;
        }
    }

    // test for end of line
    if (c) {
        // test for alphanumeric token
        if (isalnum(c) || c == '_') {
            while (isalnum(c) || c == '_' || c == '$') {
                *word++ = toupper(c);
                c = *++linePtr;
            }
            *word = 0;
            return -1;
        } else {
            word[0] = c;
            word[1] = 0;
            linePtr++;
            return c;
        }
    }

    return 0;
}


// ==============================================================
// text I/O


// reads the next line from a text file
//      line     - the buffer to read into
//      max      - sizeof the line buffer
//      returns 0 if EOF, 1 if a line was read
int ReadLine(FILE *file, char *line, int max)
{
    linenum++;          // increment line number counter
    linePtr = line;     // set GetWord() pointer to start of line
    *line = 0;          // store initial null terminator
    int len = 0;

    while (1) {
        int c = fgetc(file);        // get next char from file
        switch (c) {
            case EOF:
                if (len == 0) {
                    return 0;       // if line at end of file is empty, return EOF
                }
                // else fall through...
            case '\n':
                return 1;           // end of line

            case '\r':
                break;              // ignore the return character of CRLF

            default:
                if (max > 1) {      // if there is space left in the line buffer
                    *line++ = c;    // add character to line
                    *line   = 0;    // ensure line is null terminated
                    max--;          // update size counters
                    len++;
                }
                break;
        }
    }

//  return 1; // loop never exits to here
}


int GetSym(void)
{
    curToken = GetWord(word);
    if (curToken == 0) {
        int i = ReadLine(source, line, sizeof(line));
        while (i) {
            curToken = GetWord(word);
            if (curToken) {
                break;
            }

            i = ReadLine(source, line, sizeof(line));
        }
    }

    return curToken;
}


int GetNum(void)
{
    if (GetSym() == -1) {
        return EvalNum(word);
    }
    Error("Number expected");
    return 0;
}


#if 0 // not used
void Expect(char *expected)
{
    Str255 s;

    GetSym();
    if (strcmp(word, expected) != 0) {
        sprintf(s, "\"%s\" expected", expected);
        Error(s);
    }
}
#endif


// returns TRUE if 's' == 'word'
bool kwd(char *word, char *s)
{
    return (strcmp(word, s) == 0);
}


void EatLine(void)
{
    if (curToken != ';') {
        int token = 1;
        while (token != ';' && token != 0) {
            token = GetSym();
        }
    }
}


// ==============================================================


int FindPinName(char *name)
{
    for (int i = 0; i <= chip->phantomPins; i++) {
        if (pinName[i] && strcmp(pinName[i], word) == 0) {
            return i;
        }
    }
    return 0;
}


void DoChip(void)
{
    Str255  s;

    if (chip) {
        Error("Chip type already selected");
        return;
    }

    int token = GetSym();
    if (token == -1) {
        for (int i = 0; chipList[i]; i++) {
             if (strcmp(word, chipList[i]->chipName) == 0) {
                 chip = chipList[i];
                 break;
             }
        }
    }

    if (!chip) {
        sprintf(s, "Unknown chip type '%s'", word);
        Error(s);
        return;
    }

    // initialize writeable outOE
    for (int i = 0; i <= chip->phantomPins; i++) {
        hasOE[i] = chip->pins[i].outOE == has_OE;
    }

    // set all Product Term Disable fuses TRUE to disable PTD function
    if (chip->ptdBase >= 0) {
        for (int i = chip->ptdBase; i < chip->ptdBase + chip->numRows; i++) {
            fuseMap[i] = true;
        }
    }

    // handle chip mode fuse map settings (hopefully nothing needs fuse 0 set!)
    if (chip->addFuses) {
        for (int i = 0; chip->addFuses[i]; i++) {
            int fuse = chip->addFuses[i];
            fuseMap[abs(fuse)] = (fuse > 0);
        }
    }

    // build col2pin
    for (int i = 0; i < chip->devPins; i++) {
        if (chip->pins[i].inFuse >= 0) {
            col2pin[chip->pins[i].inFuse] = i;
        }
    }
}


void DoPin(void)
{
    int token;
    Str255 s;

    if (!chip) {
        Error("chip type not declared!");
        exit(1);
    }

    int pin = GetNum();
    if (pin < 1 || pin > chip->phantomPins) {
        sprintf(s, "invalid pin number %d", pin);
        Error(s);
        return;
    }

    if (pinName[pin]) {
        sprintf(s, "pin %d already named '%s'", pin, pinName[pin]);
        Error(s);
        return;
    }

    pinPhase[pin] = 1;  // default to active-high
    token = GetSym();
    if (token == '!' || token == '/') {
        pinPhase[pin] = 0;
        token = GetSym();
    }
    if (token != -1 || !isalpha(word[0])) {
        Error("pin name expected");
    } else {
        int i;
        if ((i = FindPinName(word)) != 0) {
            sprintf(s, "pin %d name '%s' already used for pin %d", pin, pinName[i], i);
            Error(s);
            return;
        }
        else {
            pinName[pin] = newstring(word);
        }
    }
    //printf("DoPin: pin = %2d, phase = %d, name = '%s'\n", pin, pinPhase[pin], pinName[pin]);
}


void FuseEqn(int pin, bool pinOE, int row, int lastRow)
{
    int token;
    Str255 s;

    //printf("fuseeqn: row=%d, lastRow=%d\n", row, lastRow);

    token = GetSym();
    if (kwd(word, "1")) {
        // disconnect all input terms because a zero-input AND is always = 1
        for (int i = row; i < row + chip->numCols; i++) {
            fuseMap[i] = true;
        }
        return;
    }
    else if (kwd(word, "0")) {
        // connect all input terms because A & !A always = 0
        for (int i = row; i < row + chip->numCols; i++) {
            fuseMap[i] = false;
        }
        return;
    }

    bool newrow = true;
    while (token == '!' || token == '/' || token == -1) {
        // initialize new row to all fuses set
        if (newrow) {
            if (row >= lastRow) {
                if (pinOE) {
                    sprintf(s, "too many terms for output pin %d (%s.OE)", pin, pinName[pin]);
                } else {
                    sprintf(s, "too many terms for output pin %d (%s)", pin, pinName[pin]);
                }
                Error(s);
                return;
            }
            for (int i = row; i < row + chip->numCols; i++) {
                fuseMap[i] = true;
            }
            newrow = false;
        }

        bool inputPhase = true;
        if (token == '!' || token == '/') {
            inputPhase = false;
            token = GetSym();
        }
        int input = FindPinName(word);
        if (input == 0) {
            sprintf(s, "invalid pin '%s'", word);
            Error(s);
            return;
        } else {
            int col = chip->pins[input].inFuse;
            if (col < 0) {
                sprintf(s, "pin '%s' is not an input pin", word);
                Error(s);
                return;
            }

            if (inputPhase != pinPhase[input]) {
                col ^= 1;
            }

            if (!fuseMap[row + col]) {
                sprintf(s, "input '%s' has already been used", word);
                Warning(s);
            }
            fuseMap[row + col] = false;

            // look for operator or end of statement
            token = GetSym();
            if (token == '|' || token == '+') {
                row = row + chip->numCols;
                newrow = true;
            }
            if (token == '&' || token == '*' ||
                token == '|' || token == '+') {
                // get next symbol
                token = GetSym();
            }
        }
    }
}

void DoEqn(int pin, bool phase)
{
    int     token;
    Str255  s;

    if (!chip) {
        Error("chip type not declared!");
        exit(1);
    }

    // make sure it's an output pin
    int row = chip->pins[pin].outFuse;
    int lastRow = row + chip->numCols * chip->pins[pin].outRows;
    if (row < 0) {
        Error("left side of equation must be an output pin");
        return;
    }

    //printf("DoEqn: pin = %d, phase = %d\n", pin, phase);

    bool mainterm = true;
    bool pinOE = false;

    token = GetSym();
    if (token == '.') {
        mainterm = false;
        token = GetSym();
        if (kwd(word, "OE")) {
            if (chip->pins[pin].outOE <= 0) {
                Error("pin does not have OE term");
                return;
            }
            if (oeDef[pin]) {
                Error("pin OE term already defined");
                return;
            }

            // check if optional OE term (16V8R / 20V8R)
            if (chip->pins[pin].outOE == opt_OE) {
                if (pinMode[pin] & M_REG) {
                    sprintf(s, "Can't use .OE term on registered output pin %d (%s)",
                               pin, pinName[pin]);
                    Error(s);
                    return;
                }

                // mark pin as used with OE
                hasOE[pin] = true;

                // block move other terms down in case they were already defined
                // note: really should check if last row wasn't blank because it'll be lost
                for (int i = lastRow - 1; i >= row + chip->numCols; i--) {
                    fuseMap[i] = fuseMap[i - chip->numCols];
                }
                // clear out OE row
                for (int i = row; i < row + chip->numCols; i++) {
                    fuseMap[i] = 0;
                }
            }

            lastRow = row + chip->numCols;
            oeDef[pin] = true;
            pinOE = true;
            token = GetSym();
        } else {
            Error("unknown control pin");
            return;
        }
    } else {
        // skip first row if output enable
//        if (chip->pins[pin].outOE == has_OE) {
        if (hasOE[pin]) {
            row = row + chip->numCols;
        }
        lastRow = chip->pins[pin].outFuse + chip->numCols * chip->pins[pin].outRows;

        if (outDef[pin]) {
            sprintf(s, "pin %d (%s) already defined", pin, pinName[pin]);
            Error(s);
            return;
        }
        outDef[pin] = true;

        if (phase != pinPhase[pin]) {
            if (chip->pins[pin].pinModes & M_XOR) {
                pinMode[pin] |=  M_XOR;
                pinMode[pin] &= ~M_INPUT;
                //printf("pin %d mode is now %d\n", pin, pinMode[pin]);
            } else {
                sprintf(s, "pin %d (%s) does not support output XOR", pin, pinName[pin]);
                Error(s);
                return;
            }
        }
    }

    // look for assignment operator
    switch (token) {
        case ':':
            token = GetSym();
            if (token != '=') {
                Error("invalid operator in equation");
                return;
            }
            if (!mainterm || !(chip->pins[pin].pinModes & M_REG)) {
                Error("can't make registered assignment");
                return;
            }
            if (chip->pins[0].outOE == opt_OE && oeDef[pin]) {
                // 16V8R / 20V8R can't use OE terms on registered outputs
                sprintf(s, "Can't use .OE term on registered output pin %d (%s)",
                           pin, pinName[pin]);
                Error(s);
                return;
            }

            // operator = registered assignment
            pinMode[pin] |=  M_REG;
            pinMode[pin] &= ~M_INPUT;
            //printf("pin %d mode is now %d\n", pin, pinMode[pin]);
            break;

        case '=':
            // operator = assignment
            pinMode[pin] &= ~M_INPUT;
            //printf("pin %d mode is now %d\n", pin, pinMode[pin]);
            break;

        default:
            Error("invalid operator in equation");
            return;
    }

    FuseEqn(pin, pinOE, row, lastRow);
}


// Before dumping the fusemap, check for registered feedback terms.
// Registered feedback terms on all of 16V8/20V8/22V10 are inverted, so
// look for any input terms that use them. If found, just swap the bits.

void DoFeedbackTerms(void)
{
    for (int pin = 1; pin <= chip->devPins; pin++) {
        if (pinMode[pin] & M_REG) {
            for (int row = 0; row < chip->numRows; row++) {
                int fuse = chip->pins[pin].inFuse + row * chip->numCols;
                if (fuseMap[fuse] ^ fuseMap[fuse+1]) {
                    fuseMap[fuse  ] = !fuseMap[fuse  ];
                    fuseMap[fuse+1] = !fuseMap[fuse+1];
                }
            }
        }
    }
}


bool AddTest(char *test)
{
    if (numTests >= MAXTESTS) {
        Error("too many test vectors");
        return true;
    } else {
        tests[numTests] = newstring(test);
        if (tests[numTests] == NULL) {
            Error("can't allocate string for test vector");
            return true;
        }
        numTests++;
    }
    return false;
}


bool IsFeedbackPin(int pin)
{
    if (chip->pins[pin].outFuse < 0) {
        return 0;         // not a feedback pin if no row fuse for pin
    }
    if (pinMode[pin] & M_REG) {
        return 1;     // always a feedback pin if registered
    }
    if (pinMode[pin] & M_INPUT) {
        return 0;   // never a feedback term if input-only
    }

    // if any pair of input fuses is 01 or 10, return 1
    for (int fuse = chip->pins[pin].outFuse;
             fuse < chip->pins[pin].outFuse + chip->numCols * chip->pins[pin].outRows;
             fuse = fuse + 2) {
        if (fuseMap[fuse] ^ fuseMap[fuse+1]) {
            printf("input term at fuse %d\n", fuse);
            return 1;
        }
    }

    // otherwise it's probably not a feedback term
    return 0;
}


void TestPins(void)
{
    Str255 s;
    char test[MAXPINS+1];
    int token;
    int numInp, numOut;
    int pinType[MAXPINS+1];
    int pinVal[MAXPINS+1];
    int phase[MAXPINS+1];

    // values for pinType
    enum {
        nullPin,
        inpPin,
        outPin
    };

    //printf("--- TestPins ---\n");

    for (int i = 0; i <= chip->devPins; i++) {
        test[i]    = 'N';
        pinType[i] = nullPin;
        phase[i]   = true;
    }
    test[chip->devPins] = 0;

    numInp = 0;
    numOut = 0;

    // first get the names of the output pins
    token = curToken;
    if (token == ';') {
        Error("no test pins specified");
        return;
    }
    while (token != 0 && token != ';') {
        int pin = FindPinName(word);
        if (pin < 1 || chip->pins[pin].outFuse < 0 
                    || (pinMode[pin] & M_REG)
                    || (pinMode[pin] & M_INPUT)) {
            sprintf(s, "pin '%s' must be a non-registered output pin", word);
            Error(s);
            return;
        }
        else if (pinType[pin] == outPin) {
            Error("pin specified twice");
            return;
        }

        pinType[pin] = outPin;
        if (pinMode[pin] & M_XOR) {
            phase[pin] = false;
        } else {
            phase[pin] = true;
        }
        numOut++;

        token = GetSym();

        //printf("Test output pin %d (%s), mode=%d, pinPhase=%d, phase=%d\n", 
        //       pin, pinName[pin], pinMode[pin], pinPhase[pin], phase[pin]);
    }

    // for every output pin, find its inputs
    for (int pin = 1; pin <= chip->devPins; pin++)
        if (pinType[pin] == outPin) {
            //printf("Pin %d (%s) is an output, first row=%d, rows=%d\n", 
            //       pin, pinName[pin], hasOE[pin] ? 1 : 0, chip->pins[pin].outRows);
            for (int row = (hasOE[pin]) ? 1:0; row < chip->pins[pin].outRows; row++)
                for (int col = 0; col < chip->numCols; col = col + 2) {
                    int fuse = chip->pins[pin].outFuse + row * chip->numCols + col;
                    if (fuseMap[fuse] ^ fuseMap[fuse+1]) {
                        int inPin = col2pin[col];
                        if (inPin < 1 || inPin > chip->devPins) {
                            // input column not found
                            sprintf(s, "Output pin %d (%s), input from column %d is not a valid pin",
                                       pin, pinName[pin], col);
                            Error(s);
                            return;
                        }
                        if (IsFeedbackPin(inPin)) {
                            // feedback pins can be a real mess, so I'm not handling them now
                            sprintf(s, "Output pin %d (%s), input from pin %d (%s) is a feedback term",
                                       pin, pinName[pin], inPin, pinName[inPin]);
                            Error(s);
                            return;
                        }
                        if (pinType[inPin] == nullPin) {
                            //printf("  input from pin %d (%s)\n", inPin, pinName[inPin]);
                            pinType[inPin] = inpPin;
                            //phase[inPin]   = pinPhase[i];
                            ////if (fuseMap[fuse+1] == 0) { // ^ !fuseMap[fuse+1]
                            ////    phase[inPin] = false;
                            ////}
                            numInp++;
                        }
                    }
                }
        }
/*
    // create template from input and output pins
    for (int pin = 1; pin <= chip->devPins; pin++) {
        switch (pinType[pin]) {
            default:
            case nullPin:
                test[pin-1] = 'N';
                break;

            case inpPin:
                test[pin-1] = '0';
                break;

            case outPin:
                test[pin-1] = 'L';
                break;
        }
    }
    printf("%d input terms, %d output terms, template = '%s'\n", numInp, numOut, test);
*/
    // iterate all combinations of inputs
    for (int combo = 0; combo < (1<<numInp); combo++) {
        int j = 1;
        for (int pin = chip->devPins; pin > 0; pin--) {
            switch (pinType[pin]) {
                default:
                case nullPin:
                    test[pin-1] = 'N';
                    break;

                case inpPin:
                    pinVal[pin] = (combo & j) != 0;
                    if (combo & j) {
                        test[pin-1] = '1';
                    } else {
                        test[pin-1] = '0';
                    }
                    j = j << 1;
                    break;

                case outPin:
                    test[pin-1] = '?';
                    break;
            }
        }

        // simulate output pins
        for (int pin = chip->devPins; pin > 0; pin--)
            if (pinType[pin] == outPin) {
                int result = 1;
                for (int row = (hasOE[pin]) ? 1 : 0; row < chip->pins[pin].outRows; row++) {
                    result = 1;
                    for (int col = 0; col < chip->numCols; col = col + 2) {
                        int fuse = chip->pins[pin].outFuse + row * chip->numCols + col;
                        //printf("%.4d: pin %d, row %d, col %d, fuse=%d: %d %d %d\n",
                        //       combo, pin, row, col, fuse, fuseMap[fuse], fuseMap[fuse+1],
                        //       fuseMap[fuse] + fuseMap[fuse+1]*2);
                        switch (fuseMap[fuse] + fuseMap[fuse+1]*2) {
                            case 0: // both terms used
                                //printf("  both terms used, result = 0\n");
                                result = 0;
                                break;

                            case 1: { // inverted term used
                                int inPin = col2pin[col];
                                //printf("  input pin %d (%s), !value=%d\n",
                                //       inPin, pinName[inPin], !pinVal[inPin]);
                                if (inPin >= 1 && inpPin <= chip->devPins) {
                                    result = !pinVal[inPin];
                                    //if (phase[inPin]) { // ^ phase[inPin]
                                    //    result = !result;
                                    //}
                                }
                                break;
                            }

                            case 2: { // non-inverted term used
                                int inPin = col2pin[col];
                                //printf("  input pin %d (%s), value=%d\n",
                                //       inPin, pinName[inPin], pinVal[inPin]);
                                if (inPin >= 1 && inPin <= chip->devPins) {
                                    result = pinVal[inPin];
                                    //if (phase[inPin]) { // ^ phase[inPin]
                                    //    result = !result;
                                    //}
                                }
                                break;
                            }

                            default:
                            case 3: // neither input used
                                //printf("  neither term used, continuing\n");
                                break;
                        }
                        //printf("  result = %d\n", result);
                        if (result == 0) {
                            break;
                        }
                    }
                    if (result == 1) {
                        break;
                    }
                }
                //printf("%.4d: pin %d (%s) phase=%d result=%d\n",
                //       combo, pin, pinName[pin], phase[pin], result);
                result ^= !phase[pin];
                if (result == 0) {
                    test[pin-1] = 'L';
                } else {
                    test[pin-1] = 'H';
                }
            }

        //printf("%.4d: %s\n", combo, test);
        if (AddTest(test)) {
            return; // could also be 'break' I guess
        }
    }
}


void DoTest(void)
{
    char test[MAXPINS+1];
    bool phase;
    char vals[4];

    int token;

    test[0] = 0;

    token = GetSym();
    if (token == '[') {
        vals[0] = '0';
        vals[1] = '1';
        vals[2] = 'C';
        vals[3] = 'K';

        int i;
        for (i = 0; i < chip->devPins; i++) {
            test[i] = 'N';
        }

        test[i] = 0;

        token = GetSym();
        while (token != ']' && token != ';' && token != 0) {
            phase = true;
            if (token == '!' || token == '/') {
                phase = false;
                token = GetSym();
            }
            if (token == -1) {
                i = FindPinName(word);
                if (i > 0) {
                    phase ^= !pinPhase[i];

                    token = GetSym();
                    if (token == '=') {
                        token = GetSym();
                        if (word[0] != 0 && word[1] == 0 &&
                            ( (vals[0]=='0' && word[0] != 'H'
                                            && word[0] != 'L') ||
                              (vals[0]!='0' && word[0] != '0'
                                            && word[0] != '1'
                                            && word[0] != 'C'
                                            && word[0] != 'K') )) {
                            if (!phase) {
                                switch (word[0]) {
                                    case '0': word[0]='1'; break;
                                    case '1': word[0]='0'; break;
                                    case 'L': word[0]='H'; break;
                                    case 'H': word[0]='L'; break;
                                    case 'C': word[0]='K'; break;
                                    case 'K': word[0]='C'; break;
                                }
                            }
                            test[i-1] = word[0];
                        } else {
                            Error("invalid pin test value");
                        }
                        token = GetSym();
                    } else {
                        test[i-1] = vals[phase];
                    }
                } else {
                    Error("unknown pin name");
                }
            } else {
                Error("pin name expected");
            }

            token = curToken;
            if (token == '-') {
                token = GetSym();
                if (token != '>') {
                    Error("'->' expected");
                }
                else if (vals[0] != '0') {
                    Error("'->' already specified");
                }
                else {
                    token = GetSym();
                    vals[0] = 'L';
                    vals[1] = 'H';
                    vals[2] = ' ';
                    vals[3] = ' ';
                }
            }
        }

        AddTest(test);
    }
    else if (FindPinName(word) > 0) {
        TestPins();
    }
    else {
        // raw JEDEC test vectors
        while (token != ';') {
            strcat(test, word);
            token = GetSym();
        }
        AddTest(test);
    }
}

bool Statement(void)
{
    int         token;
    int         pin;
    Str255      s;

    // get next token and exit if end of file
    token = GetSym();
    if (token == 0) {
        return 0;
    }

    if (token != ';') {
        // select chip type
             if (kwd(word, "CHIP")) {
             DoChip();
        }

        // declare a pin
        else if (kwd(word, "PIN" )) {
             DoPin();
        }

        // add a test vector
        else if (kwd(word, "TEST")) {
             DoTest();
        }

        // negative equation
        else if (token == '!' || token == '/') {
            token = GetSym();
            if (token != -1 || (pin = FindPinName(word)) == 0) {
                Error("invalid pin name");
            } else {
                DoEqn(pin, 0);
            }
        }

        // positive equation
        else if ((pin = FindPinName(word)) != 0) {
            DoEqn(pin, 1);
        }

        // unknown statement
        else {
            sprintf(s, "Invalid keyword '%s'", word);
            Error(s);
        }
    }

    // expect a semicolon at end of line
    if (curToken != ';') {
        GetSym();
        if (curToken != ';') {
            Error("';' expected");
        }
    }
    return 1;
}


void DoModes(void)
{
    for (int pin = 1; pin <= chip->devPins; pin++) {
        if (chip->pins[pin].outFuse >= 0) {
            int mode = chip->modes[pinMode[pin]];
            if (mode < 0) {
                if (pinMode[pin] == M_INPUT) {
                    fprintf(stderr, "pin %d (%s) invalid output mode (use OE to disable output)\n",
                                    pin, pinName[pin]);
                } else {
                    fprintf(stderr, "pin %d (%s) invalid output mode\n", pin, pinName[pin]);
                }
            } else {
                //fprintf(stderr, "pin %d output mode = %d:%d - ", pin, pinMode[pin], mode);
                for (int i = 0; i < MAXOPTS; i++) {
                    int fuse = chip->pins[pin].optFuses[i];
                    if (fuse > 0) {
                        fuseMap[fuse] = mode & (1 << i);
                        //fprintf(stderr, " fuse %d = %d", fuse, fuseMap[fuse]);
                    }
                }
                //fprintf(stderr, "\n");
            }
        }
    }
}


void DoFile(void)
{
    linenum = 0;
    linePtr = line;

    while (Statement()) {
        // loop until end of file
    }
}


// ==============================================================


void DumpFuseMap(void)
{
    int     addr, col;
    int     chksum;

    col = 0;
    addr = 0;
    chksum = 0;
    fprintf(object, "\nL%.4d\n", addr);

    while (addr < chip->devFuses) {
        if (fuseMap[addr]) {
            chksum = chksum + (1 << (addr & 7));
        }

        int i;
        for (i = 0; chip->skipLine[i] && addr != chip->skipLine[i]; i++) {
            // loop until match found
        }

        if (addr != 0 && addr == chip->skipLine[i]) {
            fprintf(object, "*\n\nL%.4d\n", addr);
            col = 0;
        }
        else if (addr < chip->numRows * chip->numCols && col == chip->numCols) {
            printf("\n");
            col = 0;
        }
        if (fuseMap[addr]) {
            printf("1");
        } else {
            printf("0");
        }
        addr++;
        col++;
    }

    fprintf(object, "*\n");
    fprintf(object, "\nC%.4X*\n\n", chksum);   // print fusemap checksum
}


void DumpTests()
{
    if (numTests) {
        fprintf(object, "N test vectors *\n\n");

        for (int i = 0; i < numTests; i++) {
            fprintf(object, "V%.4d %s*\n", i+1, tests[i]);
        }

        fprintf(object, "\n");
    }
}


void DumpInfo(void)
{
    fprintf(object, "%c", '\x02');      // print STX character
    // print comments at top
    fprintf(object, "Chip type = %s\n", chip -> chipName);
    fprintf(object, "Source file: %s\n", cl_SrcName);
    fprintf(object, "*\n");

    fprintf(object, "QP%2d*    N %d pins * \n", chip->devPins, chip->devPins);
    fprintf(object, "QF%.4d*  N %d fuses *\n", chip->devFuses, chip->devFuses);
    fprintf(object, "QV%.4d*  N %d test vectors *\n", numTests, numTests);
    fprintf(object, "F%d*      N default fuse state *\n", defFuse);
    if (secFuse) {
        fprintf(object, "G1*      N blow the security fuse *\n");
    } else {
        fprintf(object, "G0*      N don't blow the security fuse *\n");
    }

    DumpFuseMap();

    DumpTests();

    fprintf(object, "%c%.4X\n", 3, 0);  // print ETX character and bogus text checksum
}


// ==============================================================


void Initialize(void)
{
    chip        = NULL;

    numTests    = 0;
    defFuse     = 0;
    secFuse     = false;

    for (int i = 0; i < MAXFUSES; i++) {
        fuseMap[i] = false;
    }

    for (int i = 0; i < MAXPINS; i++) {
        pinName [i] =  NULL;
        pinPhase[i] =     1;
        hasOE   [i] = false;
        nextRow [i] =     0;
        pinMode [i] =     0;
        outDef  [i] = false;
        oeDef   [i] = false;
        pinMode [i] = M_INPUT;
    }

    for (int i = 0; i < MAXCOLS; i++) {
        col2pin[i] = 0;
    }

    memset(tests, 0, sizeof(tests));
}


// ==============================================================
// initialization and parameters


void usage(void)
{
    fprintf(stderr, "%s version %s\n", versionName, VERSION);
    fprintf(stderr, "\n");
    fprintf(stderr, "Usage:\n");
    fprintf(stderr, "    %s srcfile\n", progname);
    exit(1);
}


void getopts(int argc, char * const argv[])
{
    progname = argv[0];
    cl_SrcName [0] = 0;

    int opt;
    while ((opt = getopt(argc, argv, "?")) != -1) {
        switch (opt) {
            case '?':
            default:
                usage();
        }
    }
    argc -= optind;
    argv += optind;

    // now argc is the number of remaining arguments
    // and argv[0] is the first remaining argument

    if (argc != 1) {
        usage();
    }

    strncpy(cl_SrcName, argv[0], 255);

    // note: this won't work in *nix if there's a single-char filename in the current directory!
    if (cl_SrcName[0] == '?' && cl_SrcName[1] == 0) {
        usage();
    }
}


int main(int argc, char * const argv[])
{
    // get command line parameters
    getopts(argc, argv);

    // open the input file
    source = fopen(cl_SrcName, "r");
    if (source == NULL) {
        fprintf(stderr, "Unable to open source input file '%s'!\n", cl_SrcName);
        exit(1);
    }

    // set the output file to be stdout
    object = stdout;

    // process the input file
    Initialize();
    DoFile();
    DoModes();
    DoFeedbackTerms();
    DumpInfo();

    // close the files if they are open
    if (source) {
        fclose(source);
    }
    if (object && object != stdout) {
        fclose(object);
    }

//  return (errCount != 0);
    return 0;
}
